#region References

using System;
using System.Collections;
using System.Data;
using System.Reflection;
using Method = System.Reflection.MethodBase;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.VAL;

using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmEnterPostTransfusionInfo;

#endregion

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer></Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate></CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// Patient transfusion business object layer class
	///</summary>

	#endregion

	public class PatientTransfusion : BOL.BaseBusinessObject
	{
        //Defect 236167
        private const int TRANSFUSION_START_END_DATE_TIME_OFFSET = 5;

		#region Variables

		private DataRow _patientTransfusionDataRow;
		private DataRow _patientTransfusionCommentDataRow;

		private DataTable _dtLocations;
		private DataTable _patientTransfusionDataTable;
		private DataTable _patientTransfusionCommentDataTable;
		
		private bool _aboIncompatible;
		//
		// Comment Update: indicates whether the comment has changes so it can be saved.
		private bool _commentDirty;
		private bool _remoteStorageIndicator;

		private string _aboIncompatibleComment;

		private decimal _volume;

		private DateTime _issueDateTime;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4059"> 
		///		<ExpectedInput>existing guids</ExpectedInput>
		///		<ExpectedOutput>PatientTransfusion object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="4060"> 
		///		<ExpectedInput>non-existing guids</ExpectedInput>
		///		<ExpectedOutput>New PatientTransfusion object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// 
		/// <summary>
		/// Constructor to be used in creating or updating a Patient Transfusion record.  This 
		/// constructor will determine if this is a new or existing record and set the properties
		/// appropriately.
		/// </summary>
		public PatientTransfusion(Guid patientGuid, Guid bloodUnitGuid, bool aboIncompatible)
		{
			_aboIncompatible = aboIncompatible;
			_patientTransfusionDataTable = DAL.PatientTransfusion.GetPatientTransfusionRecord(patientGuid,
				bloodUnitGuid);
			
			if (_patientTransfusionDataTable.Rows.Count > 0) 
			{
				// BR_69.08 loaded for edit
				_patientTransfusionDataRow = _patientTransfusionDataTable.Rows[0];
				this.RowVersion = (byte[])_patientTransfusionDataTable.Rows[0][TABLE.PatientTransfusion.RowVersion];
				_patientTransfusionCommentDataTable = GetComments();
				
				if (_patientTransfusionCommentDataTable.Rows.Count>0)
				{
					_patientTransfusionCommentDataRow = LoadCommentDataRow(_patientTransfusionCommentDataTable.Rows[0]);
				}
				else
				{
					_patientTransfusionCommentDataTable = GetTransfusionCommentDataTableSchema(false);
					_patientTransfusionCommentDataRow = _patientTransfusionCommentDataTable.NewRow();
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid] = System.Guid.NewGuid();
				}
			}
			else 
			{
				MakeNew();
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.PatientGuid] = patientGuid;
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid] = bloodUnitGuid;
			}

			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LastUpdateUser] = Common.LogonUser.LogonUserName;

			if (aboIncompatible)
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LastUpdateFunctionId] = (int) Common.UpdateFunction.UC069FrmAboIncompatibleTransfusion;
			}
			else
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LastUpdateFunctionId] = (int) Common.UpdateFunction.UC069FrmEnterPostTransfusionInfo;
			}

			GetTransfusionRelatedInformation(patientGuid, bloodUnitGuid);
		}

		///<Developers>
		///	<Developer>Department of Veterans Affairs</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/18/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9192"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9193"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor to be used in creating or updating a Patient Transfusion record.  This 
		/// constructor will determine if this is a new or existing record and set the properties
		/// appropriately. 
		/// CR 2959
		/// </summary>
		public PatientTransfusion(Guid patientGuid, Guid bloodUnitGuid, string updateUserDivisionCode, string updateUserId)
		{
			_patientTransfusionDataTable = DAL.PatientTransfusion.GetPatientTransfusionRecord(patientGuid, bloodUnitGuid);
			//
			if (_patientTransfusionDataTable.Rows.Count > 0) 
			{
				// BR_69.08 loaded for edit
				_patientTransfusionDataRow = _patientTransfusionDataTable.Rows[0];
				//
				this.RowVersion = (byte[])_patientTransfusionDataTable.Rows[0][TABLE.PatientTransfusion.RowVersion];
				//
				_patientTransfusionCommentDataTable = GetComments();
				//
				if (_patientTransfusionCommentDataTable.Rows.Count > 0)
				{
					_patientTransfusionCommentDataRow = LoadCommentDataRow(_patientTransfusionCommentDataTable.Rows[0]);
				}
				else
				{
					_patientTransfusionCommentDataTable = GetTransfusionCommentDataTableSchema(false);
					_patientTransfusionCommentDataRow = _patientTransfusionCommentDataTable.NewRow();
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid] = System.Guid.NewGuid();
				}
			}
			else 
			{
				MakeNew();
				//
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.PatientGuid] = patientGuid;
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid] = bloodUnitGuid;
			}
			//
			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.DivisionCode] = updateUserDivisionCode;
			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LastUpdateUser] = updateUserId;
			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LastUpdateFunctionId] = (int) Common.UpdateFunction.UC069FrmEnterPostTransfusionInfo;
			//
			GetTransfusionRelatedInformation(patientGuid, bloodUnitGuid);
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3564"> 
		///		<ExpectedInput>bloodunitguid </ExpectedInput>
		///		<ExpectedOutput>PatientTransfusion object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="3565"> 
		///		<ExpectedInput>non existing Guid</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Just BloodUnitGuid Ctor
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public PatientTransfusion(Guid bloodUnitGuid)
		{
			_patientTransfusionDataTable = DAL.PatientTransfusion.GetPatientTransfusionRecord(bloodUnitGuid);
			
			if (_patientTransfusionDataTable.Rows.Count == 0)
			{
				throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.NoDataRowsFoundException("for selected unit").ResString);
			}
			if (_patientTransfusionDataTable.Rows.Count == 1) 
			{
				_patientTransfusionDataRow = _patientTransfusionDataTable.Rows[0];
				this.RowVersion = (byte[])_patientTransfusionDataTable.Rows[0][TABLE.PatientTransfusion.RowVersion];
				_patientTransfusionCommentDataTable = GetComments();
				
				if (_patientTransfusionCommentDataTable.Rows.Count > 0)
				{
					_patientTransfusionCommentDataRow = LoadCommentDataRow(_patientTransfusionCommentDataTable.Rows[0]);
				}
				else
				{
					_patientTransfusionCommentDataTable = GetTransfusionCommentDataTableSchema(false);
					_patientTransfusionCommentDataRow = _patientTransfusionCommentDataTable.NewRow();
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid] = System.Guid.NewGuid();
				}
			}
			else if (_patientTransfusionDataTable.Rows.Count > 1)
			{
				throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.TooManyDataRowsFoundException("for selected unit").ResString);
			}

			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LastUpdateUser] = Common.LogonUser.LogonUserName;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3566"> 
		///		<ExpectedInput>DataRow </ExpectedInput>
		///		<ExpectedOutput>PatientTransfusion object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="3567"> 
		///		<ExpectedInput>invalid datarow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="dtRow"></param>
		public PatientTransfusion(System.Data.DataRow dtRow)
		{
			this._patientTransfusionDataRow = dtRow;

			this.RowVersion = (byte [])dtRow[TABLE.PatientTransfusion.RowVersion];

		}

		#endregion

		#region Properties

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3570"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="3571"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Comment for ABO incompatible transfusions (required)
		/// </summary>
		public string AboIncompatibleComment
		{
			set
			{
				_aboIncompatibleComment = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/18/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8540"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8541"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2959
		/// </summary>
		public int ApplicationOfRecordId
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.ApplicationOfRecordId))
				{
					return 0;
				}
				//
				return Convert.ToInt32(_patientTransfusionDataRow[VbecsTables.PatientTransfusion.ApplicationOfRecordId]);
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.ApplicationOfRecordId] = value;
				//
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3585"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3586"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BloodUnitGuid
		/// </summary>
		public Guid BloodUnitGuid
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.BloodUnitGuid))
				{
					return Guid.Empty;
				}
				return (Guid) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3576"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3577"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BedsideVerificationDate
		/// </summary>
		public DateTime BedsideVerificationDate
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.BedsideVerificationDate))
				{
					return DateTime.MinValue;
				}
				return (DateTime) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.BedsideVerificationDate];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.BedsideVerificationDate] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/18/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8542"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8543"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2959
		/// </summary>
		public string BedsideVerificationUnitManualEntryReason
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.BedsideVerificationUnitManualEntryReason))
				{
					return string.Empty;
				}
				return _patientTransfusionDataRow[VbecsTables.PatientTransfusion.BedsideVerificationUnitManualEntryReason].ToString();
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.BedsideVerificationUnitManualEntryReason] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/18/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8544"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8545"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2959
		/// </summary>
		public string InterruptionComment
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.InterruptionComment))
				{
					return string.Empty;
				}
				//
				return Convert.ToString(_patientTransfusionDataRow[VbecsTables.PatientTransfusion.InterruptionComment]);
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.InterruptionComment] = value;
				//
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3583"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3584"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Original volume - need for calcuations and selection of radio buttons in UC 69 
		/// </summary>
		public decimal OriginalVolume
		{
			get
			{
				return this._volume;
			}
			set
			{
				this._volume = value;
			} 
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4067"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4068"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PatientLocation
		/// CR 2959
		/// </summary>
		public string PatientLocation
		{
			get
			{
				if ( this.ApplicationOfRecordId == Common.Utility.GetApplicationOfRecordTypeIntegerValue(Common.ApplicationOfRecordType.VBECS) )
				{
					if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.LocationIen))
					{
						return string.Empty;
					}
					else
					{
						if (_dtLocations == null) 
						{
							GetLocations();
						}
						//
						if ( _patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.LocationIen) )
						{
							return string.Empty;
						}
						else
						{
							DataRow [] drArray = _dtLocations.Select(VbecsTables.PatientTransfusion.LocationIen + " = '" + _patientTransfusionDataRow[VbecsTables.PatientTransfusion.LocationIen].ToString() + "'");
							//
							return drArray[0][DatabaseConstants.VistALinkColumnNames.LocationName].ToString();
						}
					}
				}
				else
				{
					return _patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusionLocation].ToString();
				}
			}
			set
			{
				string filter = value;
				//
				int quoteLocation = filter.IndexOf("'");
				//
				if (quoteLocation > -1)
				{
					filter = filter.Insert(quoteLocation, "'");
				}
				//
				if (_dtLocations == null) 
				{
					this.GetLocations();
				}
				//
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusionLocation] = value;
				//
				DataRow [] dtRowArray = this._dtLocations.Select(string.Concat(Common.DatabaseConstants.VistALinkColumnNames.LocationName, " = '", filter, "'"));
				//
				if (dtRowArray.Length == 1)
				{
					_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LocationIen] = dtRowArray[0][Common.DatabaseConstants.VistALinkColumnNames.LocationIen].ToString();
				}
				else
				{
					_patientTransfusionDataRow[VbecsTables.PatientTransfusion.LocationIen] = null;
				}
				// CR 2976: also check for ApplicationOfRecord not set
				if ( this.ApplicationOfRecordId == Common.Utility.GetApplicationOfRecordTypeIntegerValue(Common.ApplicationOfRecordType.VBECS) 
					|| this.ApplicationOfRecordId == Common.Utility.GetApplicationOfRecordTypeIntegerValue(Common.ApplicationOfRecordType.NotSet))  
				{
					RuleBroken( MethodBase.GetCurrentMethod().Name, PatientLocation.Trim().Length == 0 );
				}
				//
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3581"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3582"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets a comment in the case of an ABO incompatible transfusion
		/// BR_69.16
		/// </summary>
		public string PatientTransfusionComment
		{
			set
			{
				if (!this.IsNew && value.Trim() != _patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentText].ToString().Trim())
				{
					_commentDirty = true;
				}
				_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentText] = value;
			}
			get
			{
				return _patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentText].ToString();
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3568"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="3569"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Patient Transfusion GUID
		/// </summary>
		public Guid PatientTransfusionGuid
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.PatientTransfusionGuid))
				{
					return Guid.Empty;
				}
				return (Guid) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.PatientTransfusionGuid];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.PatientTransfusionGuid] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4074"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3578"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Transfusion reaction indicator
		/// </summary>
		public bool ReactionIndicator
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.ReactionIndicator))
				{
					return false;
				}
				return  (bool) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.ReactionIndicator];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.ReactionIndicator] = value;

				// Just have to turn off this BR when it's set (required field)
				RuleBroken(MethodBase.GetCurrentMethod().Name, false );

				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3579"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3580"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Transfusion reaction indicator
		/// </summary>
		public string ReactionSymptoms
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.ReactionSymptoms))
				{
					return string.Empty;
				}
				return  (string) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.ReactionSymptoms];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.ReactionSymptoms] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4072"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4073"> 
		///		<ExpectedInput>int.MinValue w/ TransfusedInterrupted = true</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Transfused volume.  The tests for th
		/// </summary>
		public int TransfusedVolume
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.TransfusedVolume))
				{
					return int.MinValue;
				}
				return  Convert.ToInt32(_patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusedVolume]);
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusedVolume] = value;

				//CR 2602 Fixed rule for volume being less than 0
				if (TransfusionInterrupted == true && (TransfusedVolume <= 0))
					RuleBroken(MethodBase.GetCurrentMethod().Name, (TransfusionInterrupted == true && (TransfusedVolume <= 0) ) );
					//CR 2602 Adding broken rule when volume exceeds maximum volume
				else if (TransfusedVolume > OriginalVolume)
					RuleBroken(MethodBase.GetCurrentMethod().Name, (TransfusedVolume > OriginalVolume) ) ;
				else
					RuleBroken(MethodBase.GetCurrentMethod().Name, false) ;

				this.IsDirty = true;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3572"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="3573"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Transfusionist #1
		/// </summary>
		public String Transfusionist1Id
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.Transfusionist1Id))
				{
					return string.Empty;
				}
				return (String) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.Transfusionist1Id];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.Transfusionist1Id] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3574"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="3575"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Transfusionist #2
		/// </summary>
		public String Transfusionist2Id
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.Transfusionist2Id))
				{
					return string.Empty;
				}
				return (String) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.Transfusionist2Id];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.Transfusionist2Id] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4069"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4070"> 
		///		<ExpectedInput>true w/ int.MinValue volume</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Transfusion indicator
		/// </summary>
		public bool TransfusionInterrupted
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.InterruptionIndicator))
				{
					return false;
				}
				return (bool) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.InterruptionIndicator];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.InterruptionIndicator] = value;

				// Just have to turn off this BR when it's set (required field)
				RuleBroken(MethodBase.GetCurrentMethod().Name, false );
				
				//CR 2602 Fixed rule for volume being less than 0
				RuleBroken(BROKEN.TransfusedVolume,	(value && TransfusedVolume <= 0) );

				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4061"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4062"> 
		///		<ExpectedInput>Invalid DateTime</ExpectedInput>
		///		<ExpectedOutput>Broken Rules</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Transfusion start date and time
		/// </summary>
		public DateTime TransfusionStartDateTime
		{
			get
			{
				if (_patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.TransfusionStartDateTime))
				{
					return DateTime.MinValue;
				}
				return (DateTime) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusionStartDateTime];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusionStartDateTime] = value;
					
				// BR_69.06
				// CR 2500, 2984: ignore seconds when comparing dates
				RuleBroken(MethodBase.GetCurrentMethod().Name, 
				DateTime.Compare(
				TransfusionStartDateTime, 
				DateTime.Parse(_issueDateTime.ToString(Common.VBECSDateTime.DateTimeFormat))) < 0 );

				// BR_69.06 also
				RuleBroken(BROKEN.TransfusionEndDateTime, (TransfusionStartDateTime > TransfusionEndDateTime) );

				// BR_69.08
                RuleBroken(BROKEN.TransfusionStartDateTime3, (TransfusionStartDateTime > VBECSDateTime.GetDivisionCurrentDateTime().AddMinutes(TRANSFUSION_START_END_DATE_TIME_OFFSET)));

				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4065"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4066"> 
		///		<ExpectedInput>Start DateTime that is later (invalid) than the end DateTime for both
		///		a new Patient Transfusion and an edit</ExpectedInput>
		///		<ExpectedOutput>Broken Rules</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Transfusion end date and time
		/// </summary>
		public DateTime TransfusionEndDateTime
		{
			get
			{
				if ( _patientTransfusionDataRow.IsNull(VbecsTables.PatientTransfusion.TransfusionEndDateTime) )
				{
					return DateTime.MinValue;
				}
				return (DateTime) _patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusionEndDateTime];
			}
			set
			{
				_patientTransfusionDataRow[VbecsTables.PatientTransfusion.TransfusionEndDateTime] = value;

				// CR 2984
				// BR_69.06
				RuleBroken( MethodBase.GetCurrentMethod().Name, 
					(TransfusionEndDateTime < _issueDateTime || TransfusionEndDateTime < TransfusionStartDateTime) );

				// BR_69.08
                RuleBroken(BROKEN.TransfusionEndDateTime2, (TransfusionEndDateTime > VBECSDateTime.GetDivisionCurrentDateTime().AddMinutes(TRANSFUSION_START_END_DATE_TIME_OFFSET)));

				this.IsDirty = true;
			}
		}

		#endregion

		#region Methods

		/// <summary>
		/// Gets some Comments.  That's all.
		/// </summary>
		/// <returns></returns>
		private DataTable GetComments()
		{
			return DAL.PatientTransfusion.GetTransfusionComment((System.Guid)_patientTransfusionDataRow[TABLE.PatientTransfusion.PatientTransfusionGuid]);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4050"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList with a count > 0</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="4051"> 
		///		<ExpectedInput>None, but with VistALink brought down</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retieves hospital locations
		/// </summary>
		/// <returns></returns>
		public ArrayList GetLocations()
		{
			ArrayList locations = new ArrayList();

			if (_dtLocations == null)
			{
				// get hospital locations
				DataSet dsLocations = DAL.VAL.HospitalLocation.GetHospitalLocationsByDivision(Common.LogonUser.LogonUserDivisionCode.Trim());
				if (dsLocations.Tables.Count==1)
				{
					_dtLocations = dsLocations.Tables[0];
				}
			}
				
			foreach (DataRow dtRow in _dtLocations.Rows)
			{
				locations.Add(dtRow[Common.DatabaseConstants.VistALinkColumnNames.LocationName].ToString());
			}
				
			return locations;
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4044"> 
		///		<ExpectedInput>A patient (GUID) that has an issued unit record.</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4045"> 
		///		<ExpectedInput>A patient (GUID) with no associated issued units.</ExpectedInput>
		///		<ExpectedOutput>ArrayList with a count == 0</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="4096"> 
		///		<ExpectedInput>Invalid patient Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns an ArrayList of units that have been issued or presumed transfused for the 
		/// patient.
		/// BR_69.02
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static ArrayList GetIssuedAndTransfusedUnitsForPatient(Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patientGuid").ResString);
			}

			ArrayList units = new ArrayList();
			DataTable dtIssuedUnits = DAL.PatientTransfusion.GetIssuedAndTransfusedUnitsForPatient(patientGuid);
			foreach (DataRow dtRow in dtIssuedUnits.Rows)
			{
				Hashtable ht = new Hashtable();
				ht.Add(VbecsTables.IssuedUnit.BloodUnitGuid, (Guid) dtRow[VbecsTables.IssuedUnit.BloodUnitGuid]);
				//For ABO Incompatible units, there may not be an issue record!
				if (!dtRow.IsNull(VbecsTables.IssuedUnit.IssuedUnitGuid))
				{
					ht.Add(VbecsTables.IssuedUnit.IssuedUnitGuid, (Guid) dtRow[VbecsTables.IssuedUnit.IssuedUnitGuid]);
				}
				else
				{
					ht.Add(VbecsTables.IssuedUnit.IssuedUnitGuid, Guid.Empty);
				}
				//For ABO Incompatible units, there may not be an issue record!
				if (!dtRow.IsNull(VbecsTables.IssuedUnit.IssueTechId))
				{
					ht.Add(VbecsTables.IssuedUnit.IssueTechId, (string) dtRow[VbecsTables.IssuedUnit.IssueTechId]);
				}
				else
				{
					ht.Add(VbecsTables.IssuedUnit.IssueTechId, "");
				}
				ht.Add(VbecsTables.BloodUnit.EyeReadableUnitId, (string) dtRow[VbecsTables.BloodUnit.EyeReadableUnitId]);
				ht.Add(VbecsTables.BloodProduct.ProductShortName, (string) dtRow[VbecsTables.BloodProduct.ProductShortName]);
				ht.Add(VbecsTables.BloodUnit.UnitProductCode, (string) dtRow[VbecsTables.BloodUnit.UnitProductCode]);
				ht.Add(DatabaseConstants.ArtificialColumnNames.ABORh, (string) dtRow[DatabaseConstants.ArtificialColumnNames.ABORh]);
				ht.Add(VbecsTables.BloodUnitMedia.UnitExpirationDate, (DateTime) dtRow[VbecsTables.BloodUnitMedia.UnitExpirationDate]);
				//For ABO Incompatible units, there may not be an issue record!
				if (!dtRow.IsNull(VbecsTables.IssuedUnit.IssueDateTime))
				{
					ht.Add(VbecsTables.IssuedUnit.IssueDateTime, (DateTime) dtRow[VbecsTables.IssuedUnit.IssueDateTime]);
				}
				else
				{
					if(dtRow.IsNull(VbecsTables.PatientTransfusion.TransfusionEndDateTime))
					{
						ht.Add(VbecsTables.IssuedUnit.IssueDateTime, System.DateTime.MinValue);
					}
					else 
					{
						ht.Add(VbecsTables.IssuedUnit.IssueDateTime, (DateTime) dtRow[VbecsTables.PatientTransfusion.TransfusionEndDateTime]);
					}

				}
				ht.Add(VbecsTables.BloodUnitStatusCodeCurrent.UnitStatusCode, Common.Utility.GetUnitStatusCodeFromString( (string) dtRow[VbecsTables.BloodUnitStatusCodeCurrent.UnitStatusCode] ));
				ht.Add(VbecsTables.BloodUnit.OriginalVolume, Convert.ToInt32( dtRow[VbecsTables.BloodUnit.OriginalVolume] ));
				if(dtRow.IsNull(VbecsTables.PatientTransfusion.TransfusionEndDateTime))
				{
					ht.Add(VbecsTables.PatientTransfusion.TransfusionEndDateTime, System.DateTime.MinValue);
				}
				else 
				{
					ht.Add(VbecsTables.PatientTransfusion.TransfusionEndDateTime, (DateTime) dtRow[VbecsTables.PatientTransfusion.TransfusionEndDateTime]);
				}
				ht.Add(VbecsTables.BloodUnitStatus.BloodUnitStatusGuid, (Guid)dtRow[VbecsTables.BloodUnitStatus.BloodUnitStatusGuid]);
				ht.Add(VbecsTables.IssuedUnit.RemoteStorageIndicator, Convert.ToBoolean( dtRow[VbecsTables.IssuedUnit.RemoteStorageIndicator]) );
				ht.Add(VbecsTables.ComponentClassParameter.MaximumTransfusionTime, Convert.ToDouble( dtRow[VbecsTables.ComponentClassParameter.MaximumTransfusionTime]) );
				ht.Add(VbecsTables.BloodUnit.UnitDivisionId, (string) dtRow[Common.VbecsTables.BloodUnit.UnitDivisionId] );
				ht.Add(VbecsTables.DonationType.DonationTypeCode, (string) dtRow[Common.VbecsTables.DonationType.DonationTypeCode] );
				ht.Add(VbecsTables.BloodUnit.LabelTypeCode, Convert.ToChar(dtRow[Common.VbecsTables.BloodUnit.LabelTypeCode]) );
				//
				// CR 2959
				ht.Add(VbecsTables.PatientTransfusion.ApplicationOfRecordId, dtRow[VbecsTables.PatientTransfusion.ApplicationOfRecordId] );
				//
				units.Add(ht);
			}
			return units;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4467"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4468"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns an array list of blood units that are currently transfused (or persumed transfused)
		/// against the given patient specimen.
		/// </summary>
		/// <param name="patientSpecimenGuid">Patient specimen unique identifier</param>
		/// <returns>Array list of blood unit objects</returns>
		public static ArrayList GetTransfusionHistoryForPatientSpecimen(Guid patientSpecimenGuid)
		{
			if (patientSpecimenGuid==Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Patient specimen unique identifier").ResString);
			}
			//
			ArrayList transfusions = new ArrayList();
			//
			DataTable dtTransfusions = DAL.PatientTransfusion.GetTransfusionHistoryForPatientSpecimen(patientSpecimenGuid);
			//
			foreach(DataRow dr in dtTransfusions.Rows)
			{
				transfusions.Add(new BOL.BloodUnit((Guid)dr[TABLE.PatientTransfusion.BloodUnitGuid]));
			}
			//
			return transfusions;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4077"> 
		///		<ExpectedInput>Saves a valid Patient Transfusion</ExpectedInput>
		///		<ExpectedOutput>successful save</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4078"> 
		///		<ExpectedInput>Saves an invalid Patient Transfusion (Broken Rule count > 0)</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2959
		/// </summary>
		/// <param name="workloadProcessId"></param>
		/// <returns></returns>
		public bool Save(Common.WorkloadProcessID workloadProcessId)
		{
			bool successIndicator = false;
			System.Data.DataTable dtWorkload = null;
			//
			if (this.BrokenRulesCount>0)
			{
				throw new BOL.BusinessObjectException("Save not possible - Not all rules are satisfied!");
			}
			//
			DataTable dtExceptionsAboIncompatible = null;
			//
			// CR 2959
			DataTable dtExceptionsElectronicTransfusion = null;
			//
			if (IsNew)
			{
				_patientTransfusionDataTable.Rows.Add(_patientTransfusionDataRow);
				//CR 3102 apparently we dont want to generate workload for transfusions coming from BCE COTS and
				//BCE services call this existing method
				if(this.ApplicationOfRecordId != Common.Utility.GetApplicationOfRecordTypeIntegerValue(Common.ApplicationOfRecordType.BCE_COTS))
				{
					Guid [] bloodUnitGuids = new Guid[1];

					// Get the guid array for the workload generation function
					bloodUnitGuids[0] = (System.Guid)this._patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid];


					// generate the datatable of workload event data for this process
					dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData
						(
						workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false
						);
					//
					if (_aboIncompatible)
					{
						Common.Utility.AppendLastUpdateInformation(dtWorkload, Common.UpdateFunction.UC069FrmAboIncompatibleTransfusion);
					}
					else
					{
						Common.Utility.AppendLastUpdateInformation(dtWorkload, Common.UpdateFunction.UC069FrmEnterPostTransfusionInfo);
					}
				}

				// Comment
				if ( _patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentText].ToString().Trim().Length > 0 )
				{
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid]= Guid.NewGuid();
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionGuid]= this.PatientTransfusionGuid;
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.CommentCategoryCode]= "TP";
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.DivisionCode]= Common.LogonUser.LogonUserDivisionCode;
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.LastUpdateUser]= Common.LogonUser.LogonUserName;
					//
					if (_aboIncompatible)
					{
						_patientTransfusionCommentDataRow[VbecsTables.BloodUnitStatus.LastUpdateFunctionId] = (int)Common.UpdateFunction.UC069FrmAboIncompatibleTransfusion;
					}
					else
					{
						_patientTransfusionCommentDataRow[VbecsTables.BloodUnitStatus.LastUpdateFunctionId] = (int)Common.UpdateFunction.UC069FrmEnterPostTransfusionInfo;
					}
					_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
					_patientTransfusionCommentDataTable.Rows.Add(_patientTransfusionCommentDataRow);
				}
				
				if (_aboIncompatible)
				{
					// generate exception if ABO incompatible comment was entered (required to save data)
					BOL.ExceptionReport exRep = new BOL.ExceptionReport
						(
						Common.ExceptionType.ABO_incompatible_unit_transfused, 
						_aboIncompatibleComment,
						Common.LogonUser.LogonUserName
						);
					//
					exRep.GenerateExceptionAboIncompatibleTransfusion
						(
						new BOL.Patient((Guid) this._patientTransfusionDataRow[VbecsTables.Patient.PatientGuid]),
						new BOL.BloodUnit((Guid) this._patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid])
						);
					//
					if (dtExceptionsAboIncompatible == null)
					{
						dtExceptionsAboIncompatible = exRep.ExceptionData.Table.Clone();
					}
					dtExceptionsAboIncompatible.Rows.Add(exRep.ExceptionData.ItemArray);
				}
				//
				successIndicator = DAL.PatientTransfusion.Save(_patientTransfusionDataTable, dtWorkload, _patientTransfusionCommentDataTable, dtExceptionsAboIncompatible);
			}
			else // existing record
			{
				bool newComment = false;
				// Even though the PatientTransfusionRecord is just being updated, the
				// comment could be new or an update.
				if (_commentDirty)
				{
					// The following if block code does not seem to make sense; we will always have at least one row at this point
					if (this._patientTransfusionCommentDataTable.Rows.Count == 0) 
					{
						newComment = true;
						_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.CommentCategoryCode] = "TP";
						_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
						_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionGuid] = this.PatientTransfusionGuid;
						_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.LastUpdateUser] = Common.LogonUser.LogonUserName;
						if (_aboIncompatible)
						{
							_patientTransfusionCommentDataRow[VbecsTables.BloodUnitStatus.LastUpdateFunctionId] = (int)Common.UpdateFunction.UC069FrmAboIncompatibleTransfusion;
						}
						else
						{
							_patientTransfusionCommentDataRow[VbecsTables.BloodUnitStatus.LastUpdateFunctionId] = (int)Common.UpdateFunction.UC069FrmEnterPostTransfusionInfo;
						}
						_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
						_patientTransfusionCommentDataTable.Rows.Add(_patientTransfusionCommentDataRow);
					}
					else
					{
						_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.LastUpdateUser] = Common.LogonUser.LogonUserName;
					}
				}
				else
				{
					_patientTransfusionCommentDataTable = null;
				}
				//
				if (_aboIncompatible)
				{
					// Generate exception if ABO incompatible comment was entered (required to save data)
					BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.ABO_incompatible_unit_transfused, _aboIncompatibleComment, Common.LogonUser.LogonUserName);
					exRep.GenerateExceptionAboIncompatibleTransfusion(new BOL.Patient((Guid) this._patientTransfusionDataRow[VbecsTables.Patient.PatientGuid]),
						new BOL.BloodUnit((Guid) this._patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid]));

					if (dtExceptionsAboIncompatible == null)
					{
						dtExceptionsAboIncompatible = exRep.ExceptionData.Table.Clone();
					}
					dtExceptionsAboIncompatible.Rows.Add(exRep.ExceptionData.ItemArray);
				}
				//
				// BR_69.21 Generate ElectronicTransfusion exception (CR 2959)
				if ( !this._patientTransfusionDataRow.IsNull(TABLE.PatientTransfusion.ApplicationOfRecordId) )
				{
					int applicationOfRecordId = Convert.ToInt32( _patientTransfusionDataRow[TABLE.PatientTransfusion.ApplicationOfRecordId] );
					//
					if ( applicationOfRecordId == Common.Utility.GetApplicationOfRecordTypeIntegerValue(Common.ApplicationOfRecordType.BCE_COTS) )
					{
						BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Electronic_Transfusion_Record_Update, "", Common.LogonUser.LogonUserName);
						//
						exRep.GenerateExceptionElectronicTransfusion(
							this._patientTransfusionDataRow, 
							new BOL.Patient((Guid) this._patientTransfusionDataRow[VbecsTables.Patient.PatientGuid]), 
							new BOL.BloodUnit((Guid) this._patientTransfusionDataRow[VbecsTables.PatientTransfusion.BloodUnitGuid]), 
							this.PatientTransfusionComment );

						if (dtExceptionsElectronicTransfusion == null)
						{
							dtExceptionsElectronicTransfusion = exRep.ExceptionData.Table.Clone();
						}
						dtExceptionsElectronicTransfusion.Rows.Add(exRep.ExceptionData.ItemArray);
					}
				}
				//
				successIndicator = DAL.PatientTransfusion.Update(_patientTransfusionDataTable, _patientTransfusionCommentDataTable, newComment, dtExceptionsAboIncompatible, dtExceptionsElectronicTransfusion);
			}
			//
			// Send transfusion data to Dss
			Guid patientTransfusionGuid = (System.Guid)this._patientTransfusionDataRow[TABLE.PatientTransfusion.PatientTransfusionGuid];
			//
			
			//CR 2298
			if ( VistALink.IsAvailable )
			{
				DssExtract.UpdateVistADssExtract( patientTransfusionGuid, UpdateFunction.UC069FrmEnterPostTransfusionInfo );
			}
			//
			return successIndicator;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4459"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4460"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRecentlyIssuedAndTransfusedUnitsForPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static ArrayList GetRecentlyIssuedAndTransfusedUnitsForPatient(System.Guid patientGuid)
		{
			System.Collections.ArrayList dataArray = new System.Collections.ArrayList();

			System.Data.DataTable dtUnits = DAL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatient(patientGuid);

			foreach(DataRow dr in dtUnits.Rows)
			{
				dataArray.Add(dr);
			}

			return(dataArray);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8185"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of issued and transfused units</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8186"> 
		///		<ExpectedInput>Invalid patient guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a count of issued and/or transfused units for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Count of issued and/or transfused units</returns>
		public static int GetRecentlyIssuedAndTransfusedUnitsForPatientCount(Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patientGuid").ResString);
			}
			//
			return DAL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatientCount(patientGuid);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/21/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4759"> 
		///		<ExpectedInput>patientGuid</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4760"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetTransfusedUnitsForPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns>
		/// ArrayList of BloodUnits
		/// </returns>
		public static ArrayList GetTransfusedUnitsForPatient(System.Guid patientGuid)
		{
			ArrayList unitList = null;
			//
			DataTable transfusedUnits = DAL.PatientTransfusion.GetTransfusedUnitsForPatient  
				(
				patientGuid, Common.LogonUser.LogonUserDivisionCode
				);
			//
			if	(transfusedUnits.Rows.Count > 0)
			{
				unitList = new ArrayList();

				foreach (DataRow dr in transfusedUnits.Rows)
				{
					unitList.Add(new BloodUnit(dr));
				}
			}
			//
			return unitList;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/21/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4761"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4762"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetPatientTransfusionRecords
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns>
		/// ArrayList of BloodUnits
		/// </returns>
		public static ArrayList GetPatientTransfusionRecords(System.Guid patientGuid)
		{
			ArrayList recordsList = null;
			//
			DataTable transfusedRecords = DAL.PatientTransfusion.GetPatientTransfusionRecords  
				(
				patientGuid, Common.LogonUser.LogonUserDivisionCode
				);
			//
			if	(transfusedRecords.Rows.Count > 0)
			{
				recordsList = new ArrayList();

				foreach (DataRow dr in transfusedRecords.Rows)
				{
					recordsList.Add(new BOL.PatientTransfusion(dr));
				}
			}
			//
			return recordsList;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2779"> 
		///		<ExpectedInput>Valid patient GUID</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2780"> 
		///		<ExpectedInput>Invalid patient GUID</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns boolean if patient has had a transfusion within the last 3 months
		/// Implements BR_35.05
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static bool IsTransfusionHistoryforPatient(Guid patientGuid)
		{
			//BR_35.05
			return DAL.PatientTransfusion.IsTransfusionHistoryforPatient(patientGuid);
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3591"> 
		///		<ExpectedInput>Valid patient GUID, dates</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3592"> 
		///		<ExpectedInput>Invalid patient GUID</ExpectedInput>
		///		<ExpectedOutput>DateTime.MinValue</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the datetime of the last transfusion for patient and certain date period
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static DateTime GetLastTransfusionDateTime(Guid patientGuid, DateTime startDate, DateTime endDate)
		{
			DataTable dtPatientTransfusionHistory = DAL.PatientTransfusion.GetTransfusionHistoryForPatient(patientGuid, startDate, endDate);

			if (dtPatientTransfusionHistory.Rows.Count > 0)
				return (DateTime)dtPatientTransfusionHistory.Rows[0][TABLE.PatientTransfusion.TransfusionStartDateTime];
			else
				return DateTime.MinValue;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3593"> 
		///		<ExpectedInput>Valid blood GUID</ExpectedInput>
		///		<ExpectedOutput>Patient</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3594"> 
		///		<ExpectedInput>Invalid blood GUID</ExpectedInput>
		///		<ExpectedOutput>Empty Patient</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Return transfused patient
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static BOL.Patient GetTransfusedPatient(Guid bloodUnitGuid)
		{
			DataTable dtPatient = DAL.PatientTransfusion.GetTransfusedPatient(bloodUnitGuid);

			if (dtPatient.Rows.Count > 0)
			{
				return new BOL.Patient((Guid)dtPatient.Rows[0][TABLE.Patient.PatientGuid]);
			}
			else
				return new BOL.Patient();
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3595"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3596"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}


		/// <summary>
		/// Create a new instance of PatientTransfusion
		/// </summary>
		private void MakeNew()
		{
			this.IsNew = true;
			this.IsReadOnly = false;
			//
			_patientTransfusionDataRow = _patientTransfusionDataTable.NewRow();
			_patientTransfusionCommentDataTable = GetTransfusionCommentDataTableSchema(false);
			_patientTransfusionCommentDataRow = _patientTransfusionCommentDataTable.NewRow();
			_patientTransfusionCommentDataRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid] = System.Guid.NewGuid();
			//
			this.PatientTransfusionGuid = Guid.NewGuid();
			this.LastUpdateUser = Common.LogonUser.LogonUserName;
			_patientTransfusionDataRow[VbecsTables.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}


		/// <summary>
		/// Need to get some more information about the unit
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		private void GetTransfusionRelatedInformation(Guid patientGuid, Guid bloodUnitGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patientGuid").ResString);
			}

			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
			}

			DataRow dtRow = DAL.PatientTransfusion.GetTransfusionRelatedInformation(patientGuid, bloodUnitGuid);
			// This is a new one
			if (dtRow == null)
			{
				_volume = new BOL.BloodUnit(bloodUnitGuid).OriginalVolume;
				// Default the TransfusedVolume, in case there were no interruptions
				this.TransfusedVolume = System.Convert.ToInt16(_volume);
				
				// This won't actually be used for anything.  Just set it to the minimum so broken
				// rules don't get tripped
				_issueDateTime = DateTime.MinValue;
			}
				// An existing transfusion record
			else
			{
				// Save issue date using VBECSDateTime.DateTimeFormat
				// CR 2984
				_issueDateTime = DateTime.Parse( Convert.ToDateTime(dtRow[TABLE.IssuedUnit.IssueDateTime]).ToString(Common.VBECSDateTime.DateTimeFormat) );
				//
				_volume = Convert.ToInt32( dtRow[TABLE.BloodUnit.OriginalVolume] );
				_remoteStorageIndicator = (bool)( dtRow[TABLE.IssuedUnit.RemoteStorageIndicator] );
			}
		}


		/// <summary>
		/// spInsertPatientTransfusionComment, spUpdatePatientTransfusionComment
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		private DataTable GetTransfusionCommentDataTableSchema(bool isUpdate)
		{
			DataTable dt = new DataTable(TABLE.PatientTransfusionComment.TableName);

			dt.Columns.Add(TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.PatientTransfusionComment.PatientTransfusionGuid, typeof(Guid));
			dt.Columns.Add(TABLE.PatientTransfusionComment.CommentCategoryCode, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionComment.PatientTransfusionCommentText, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionComment.DivisionCode, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionComment.LastUpdateUser, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionComment.LastUpdateFunctionId, typeof(int));
			dt.Columns.Add(TABLE.PatientTransfusionComment.RecordStatusCode, typeof(string));
			if (isUpdate)
			{
				dt.Columns.Add(TABLE.PatientTransfusionComment.RowVersion, typeof(byte[]));
			}
			return dt;
		}

		private DataRow LoadCommentDataRow(DataRow drRowFromDB)
		{
			_patientTransfusionCommentDataTable = GetTransfusionCommentDataTableSchema(true);
			DataRow drNewRow = _patientTransfusionCommentDataTable.NewRow();

			
			drNewRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid] = drRowFromDB[TABLE.PatientTransfusionComment.PatientTransfusionCommentGuid];
			drNewRow[TABLE.PatientTransfusionComment.PatientTransfusionGuid] = drRowFromDB[TABLE.PatientTransfusionComment.PatientTransfusionGuid];
			drNewRow[TABLE.PatientTransfusionComment.CommentCategoryCode] = drRowFromDB[TABLE.PatientTransfusionComment.CommentCategoryCode];
			drNewRow[TABLE.PatientTransfusionComment.PatientTransfusionCommentText] = drRowFromDB[TABLE.PatientTransfusionComment.PatientTransfusionCommentText];
			drNewRow[TABLE.PatientTransfusionComment.DivisionCode] = drRowFromDB[TABLE.PatientTransfusionComment.DivisionCode];
			drNewRow[TABLE.PatientTransfusionComment.LastUpdateFunctionId] = drRowFromDB[TABLE.PatientTransfusionComment.LastUpdateFunctionId];
			drNewRow[TABLE.PatientTransfusionComment.RecordStatusCode] = drRowFromDB[TABLE.PatientTransfusionComment.RecordStatusCode];
			drNewRow[TABLE.PatientTransfusionComment.RowVersion] = drRowFromDB[TABLE.PatientTransfusionComment.RowVersion];
			_patientTransfusionCommentDataTable.Rows.Add(drNewRow);

			return drNewRow;
		}


		/// <summary>
		/// Handler for dirty flag and attempts to modify a non-writable object
		/// </summary>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			if (this.IsReadOnly && !this.IsNew) 
			{
				throw new BusinessObjectException("IssuedUnit object is marked read-only");
			}
			this.IsDirty = true;
		}


		/// <summary>
		/// 
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			throw new BOL.BusinessObjectException("Not implemented!");
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3597"> 
		///		<ExpectedInput>Start, End date</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3598"> 
		///		<ExpectedInput>Invalid dates</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusedUnitsInDateRangeCount
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static int GetTransfusedUnitsInDateRangeCount(DateTime startDate, DateTime endDate)
		{
			return DAL.PatientTransfusion.GetTransfusedUnitsInDateRangeCount(startDate, endDate, Common.LogonUser.LogonUserDivisionCode, Guid.Empty);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3599"> 
		///		<ExpectedInput>Start, End date, patientguid</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3600"> 
		///		<ExpectedInput>Invalid dates, guid</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusedUnitsInDateRangeCount
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static int GetTransfusedUnitsInDateRangeCount(DateTime startDate, DateTime endDate, Guid patientGuid)
		{
			return DAL.PatientTransfusion.GetTransfusedUnitsInDateRangeCount(startDate, endDate, Common.LogonUser.LogonUserDivisionCode, patientGuid);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3601"> 
		///		<ExpectedInput>Start, End date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3602"> 
		///		<ExpectedInput>Invalid dates</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusedUnitsInDateRange
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static DataTable GetTransfusedUnitsInDateRange(DateTime startDate, DateTime endDate)
		{
			return DAL.PatientTransfusion.GetTransfusedUnitsInDateRange(startDate, endDate, Common.LogonUser.LogonUserDivisionCode, Guid.Empty);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3603"> 
		///		<ExpectedInput>Start, End date, patient guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3608"> 
		///		<ExpectedInput>Invalid dates, guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusedUnitsInDateRange
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetTransfusedUnitsInDateRange(DateTime startDate, DateTime endDate, Guid patientGuid)
		{
			return DAL.PatientTransfusion.GetTransfusedUnitsInDateRange(startDate, endDate, Common.LogonUser.LogonUserDivisionCode, patientGuid);
		}

		#endregion
	}
}
